/**
* \file: application_state_machine.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: application-framework
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include "application_state_machine.h"

#include "dbus_interface.h"
#include "libtestbedapp_intern.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

//----------------------------------------- constants -------------------------------------------------
const char *state_name_table[]=
{
		"TBA_STATE_INIT"			,
		"TBA_STATE_HMI_NEGOTIATION"	,
		"TBA_STATE_BACKGROUND"		,
		"TBA_STATE_FOREGROUND"		,
		"TBA_STATE_NOSERVICE"		,
		"TBA_STATE_SHUTDOWN"			,
		"TBA_STATE_ERROR"
};

//----------------------------------------- attributes ------------------------------------------------
static entering_shutdown_cb_t entering_shutdown_cb=NULL;
static waking_up_cb_t waking_up_cb=NULL;
static state_changed_cb_t state_changed_cb=NULL;
static surface_id_invalid_cb_t surface_id_invalid_cb=NULL;
static surface_id_received_cb_t surface_id_received_cb=NULL;
static void *cb_user_data;

static bool service_available;

static tba_app_state_t recent_state=__TBA_STATE_UNSET;
static tba_app_state_t current_state=__TBA_STATE_UNSET;

static unsigned int surface_id=TBA_INVALID_SURFACE_ID;

static char* error_msg=NULL;
static tba_error_code_t error_code=TBA_RESULT_OK;

//----------------------------------------- public member ---------------------------------------------
void asm_init(bool sa, const tba_callbacks_t * callbacks, void *callback_user_data)
{
	log_debug("asm - Initializing application state machine.");
	if (current_state!=__TBA_STATE_UNSET)
	{
		log_debug("asm - Application state machine already initialized. Calling deinit() first.");
		asm_deinit();
	}

	cb_user_data=callback_user_data;

	if (callbacks!=NULL)
	{
		log_debug("asm - Callbacks passed. Registering them.");
		entering_shutdown_cb=callbacks->entering_shutdown_cb;
		waking_up_cb=callbacks->waking_up_cb;
		state_changed_cb=callbacks->state_changed_cb;
		surface_id_received_cb=callbacks->surface_id_received_cb;
		surface_id_invalid_cb=callbacks->surface_id_invalid_cb;
	}

	service_available=sa;
	log_debug("asm - Application now in state: %s", asm_state_get_name(TBA_STATE_INIT));
	current_state = TBA_STATE_INIT;
}

void asm_deinit(void)
{
	log_debug("asm - Deinitializing application state machine.");
	entering_shutdown_cb=NULL;
	waking_up_cb=NULL;
	state_changed_cb=NULL;
	surface_id_received_cb=NULL;
	surface_id_invalid_cb=NULL;
	current_state = __TBA_STATE_UNSET;
	surface_id=TBA_INVALID_SURFACE_ID;
	error_code=TBA_RESULT_OK;
	if (error_msg!=NULL)
	{
		free(error_msg);
		error_msg=NULL;
	}
}

tba_app_state_t asm_get_state(void)
{
	return current_state;
}

tba_app_state_t asm_get_previous_state(void)
{
	return recent_state;
}

unsigned int asm_get_surface_id(void)
{
	return surface_id;
}

const char *asm_get_error_message(void)
{
	return error_msg;
}

tba_error_code_t asm_get_error_code(void)
{
	return error_code;
}

void asm_hmi_available_trigger(unsigned int a_surface_id)
{
	tba_app_state_t new_active_state;

	if (service_available)
		new_active_state=TBA_STATE_BACKGROUND;
	else
		new_active_state=TBA_STATE_NOSERVICE;

	log_debug("asm - HMI now connected.");

	surface_id=a_surface_id;
	if (surface_id_received_cb!=NULL)
		surface_id_received_cb(a_surface_id,cb_user_data);

	//coming from application initialization
	if (current_state==TBA_STATE_HMI_NEGOTIATION)
		asm_set_state(new_active_state);
	//coming from shutdown state (application put into shutdown state before HMI appeared)
	else if (current_state==TBA_STATE_SHUTDOWN)
	{
		recent_state=new_active_state;
		log_debug("asm - Remaining in shutdown state.");
	}
	else
		log_info("asm - HMI reappered again, which should normally not happen. We are not doing anything.");
		//HMI reappered again, we are not doing anything (shall not happen, otherwise we missed loosing the HMI)
}

void asm_hmi_lost_trigger(void)
{
	if (surface_id_received_cb!=NULL)
			surface_id_invalid_cb(surface_id,cb_user_data);
	surface_id=TBA_INVALID_SURFACE_ID;

	asm_set_state(TBA_STATE_INIT);
	recent_state=TBA_STATE_INIT;
}

void asm_service_available(void)
{
	if (service_available)
		return;

	service_available=true;

	if (current_state == TBA_STATE_NOSERVICE)
		asm_set_state(TBA_STATE_BACKGROUND);
}

void asm_service_lost(void)
{
	if (!service_available)
		return;

	service_available=false;

	if (current_state == TBA_STATE_FOREGROUND || current_state == TBA_STATE_BACKGROUND)
		asm_set_state(TBA_STATE_NOSERVICE);
}

void asm_send_to_background(void)
{
	if (current_state != TBA_STATE_FOREGROUND)
		return;
	asm_set_state(TBA_STATE_BACKGROUND);
}

void asm_send_to_foreground(void)
{
	if (current_state != TBA_STATE_BACKGROUND)
		return;
	asm_set_state(TBA_STATE_FOREGROUND);
}


void asm_enter_ERROR(const char *message, tba_error_code_t error)
{
	error_msg=strdup(message);
	error_code=error;
	current_state=TBA_STATE_ERROR;
	log_error("Application entered state error.");
	log_error("Message: %s", message);
	log_error("Error code: %d", error);
}

void asm_set_state(tba_app_state_t state)
{
	recent_state=current_state;
	current_state=state;
	if (state_changed_cb!=NULL)
		state_changed_cb(recent_state, current_state, cb_user_data);
	dbus_iface_emit_state_change_signal(current_state);
	log_debug("asm - Application entered state: %s", asm_state_get_name(state));
}

bool asm_going_to_shutdown_trigger(int arg_RequestId)
{
	bool transisition_done=true;

	log_debug("asm - Trigger to go into shutdown state received (request_id: %d)",arg_RequestId);
	//already in shutdown state?
	if (current_state!=TBA_STATE_SHUTDOWN)
	{
		if (entering_shutdown_cb!=NULL)
		{
			log_debug("asm - Calling transition callback.");
			transisition_done=entering_shutdown_cb(arg_RequestId, cb_user_data);
			log_debug("asm - Transition callback returned.");
		}

		if (transisition_done)
			asm_set_state(TBA_STATE_SHUTDOWN);
		else
			log_debug("asm - Transition postponed asynchronously. Staying in current state.");
	}

	return transisition_done;
}

bool asm_waking_up_trigger(int arg_RequestId)
{
	bool transisition_done=true;
	tba_app_state_t targeted_state=recent_state;

	log_debug("asm - Trigger to wake up received (request_id: %d)",arg_RequestId);
	//actually in shutdown state?
	if (current_state==TBA_STATE_SHUTDOWN)
	{
		if (waking_up_cb!=NULL)
		{
			log_debug("asm - Calling transition callback.");
			transisition_done=waking_up_cb(recent_state, &targeted_state, arg_RequestId, cb_user_data);
			log_debug("asm - Transition callback returned.");
		}

		assert(targeted_state==TBA_STATE_BACKGROUND || targeted_state==TBA_STATE_FOREGROUND || targeted_state==TBA_STATE_NOSERVICE
				|| targeted_state==TBA_STATE_HMI_NEGOTIATION);

		if (transisition_done)
			asm_set_state(targeted_state);
		else
			log_debug("asm - Transition postponed asynchronously. Staying in current state.");
	}

	return transisition_done;
}
//-----------------------------------------------------------------------------------------------------

//----------------------------------------- private member --------------------------------------------
